<?php
// admin_stocks.php
include($_SERVER['DOCUMENT_ROOT'] . '/config.php');

// Enable error logging but disable display
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', dirname(__DIR__) . '/logs/error.log');

// Initialize variables
$message = '';
$messageType = '';

// Handle form submission via AJAX
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    header('Content-Type: application/json');
    $response = ['success' => false, 'message' => ''];
    
    try {
        if ($_POST['action'] === 'create' || $_POST['action'] === 'update') {
            // Validate inputs
            $id = isset($_POST['id']) && is_numeric($_POST['id']) ? $_POST['id'] : null;
            $name = trim($_POST['name'] ?? '');
            $symbol = trim($_POST['symbol'] ?? '');
            $minimum = isset($_POST['minimum']) && is_numeric($_POST['minimum']) ? (float)$_POST['minimum'] : 0;
            $maximum = isset($_POST['maximum']) && is_numeric($_POST['maximum']) ? (float)$_POST['maximum'] : 0;
            $cycle = trim($_POST['cycle'] ?? '');
            $status = in_array($_POST['status'] ?? '', ['active', 'inactive']) ? $_POST['status'] : 'inactive';

            // Validation
            if (empty($name) || empty($symbol) || empty($cycle)) {
                throw new Exception("All fields are required.");
            }

            if ($minimum < 0 || $maximum < 0) {
                throw new Exception("Minimum and maximum amounts must be positive.");
            }

            if ($minimum > $maximum) {
                throw new Exception("Minimum amount cannot be greater than maximum amount.");
            }

            // Update or Insert
            if ($id && $_POST['action'] === 'update') {
                $stmt = $conn->prepare("UPDATE stocks SET name = ?, symbol = ?, minimum_amount = ?, maximum_amount = ?, cycle = ?, status = ? WHERE id = ?");
                $stmt->execute([$name, $symbol, $minimum, $maximum, $cycle, $status, $id]);
                $response['message'] = "Stock updated successfully.";
            } else {
                $stmt = $conn->prepare("INSERT INTO stocks (name, symbol, minimum_amount, maximum_amount, cycle, status) VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->execute([$name, $symbol, $minimum, $maximum, $cycle, $status]);
                $response['message'] = "Stock created successfully.";
            }
            
            $response['success'] = true;
        } elseif ($_POST['action'] === 'delete' && isset($_POST['id'])) {
            $id = is_numeric($_POST['id']) ? $_POST['id'] : 0;
            $stmt = $conn->prepare("DELETE FROM stocks WHERE id = ?");
            $stmt->execute([$id]);
            $response['success'] = true;
            $response['message'] = "Stock deleted successfully.";
        } elseif ($_POST['action'] === 'get' && isset($_POST['id'])) {
            $id = is_numeric($_POST['id']) ? $_POST['id'] : 0;
            $stmt = $conn->prepare("SELECT * FROM stocks WHERE id = ?");
            $stmt->execute([$id]);
            $stock = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($stock) {
                $response['success'] = true;
                $response['data'] = $stock;
            } else {
                $response['message'] = "Stock not found.";
            }
        }
    } catch (Exception $e) {
        $response['message'] = $e->getMessage();
    }
    
    echo json_encode($response);
    exit;
}

// Get all stocks
try {
    $stocks = $conn->query("SELECT * FROM stocks ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Fetch stocks error: " . $e->getMessage());
    $stocks = [];
    $message = "Error fetching stocks.";
    $messageType = "error";
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin - Manage Stocks</title>
    <style>
        /* General Styles */
        body { font-family: Arial, sans-serif; padding: 20px; background: #f4f4f4; line-height: 1.6; margin: 0; }
        h1 { color: #333; }
        
        /* Table Styles */
        table { width: 100%; border-collapse: collapse; margin-top: 20px; background: #fff; box-shadow: 0 1px 3px rgba(0,0,0,0.1); }
        table, th, td { border: 1px solid #ddd; }
        th, td { padding: 12px; text-align: left; }
        th { background-color: #f8f8f8; }
        tr:nth-child(even) { background-color: #f9f9f9; }
        tr:hover { background-color: #f1f1f1; }
        
        /* Button Styles */
        .btn { padding: 8px 16px; background: #006b91; color: white; border: none; border-radius: 4px; cursor: pointer; text-decoration: none; display: inline-block; margin-right: 5px; }
        .btn:hover { background: #004f6b; }
        .delete-btn { background: #d9534f; }
        .delete-btn:hover { background: #c9302c; }
        .add-btn { margin-bottom: 20px; background: #5cb85c; }
        .add-btn:hover { background: #4cae4c; }
        
        /* Form Styles */
        .form-group { margin-bottom: 15px; }
        .form-group label { display: block; margin-bottom: 5px; font-weight: bold; }
        .form-group input, .form-group select { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; box-sizing: border-box; }
        
        /* Modal Styles */
        .modal { display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.4); }
        .modal-content { background-color: #fefefe; margin: 10% auto; padding: 20px; border: 1px solid #888; width: 50%; box-shadow: 0 4px 8px rgba(0,0,0,0.1); border-radius: 5px; }
        .close { color: #aaa; float: right; font-size: 28px; font-weight: bold; cursor: pointer; }
        .close:hover { color: black; }
        
        /* Message Styles */
        #message-container { padding: 10px; margin: 20px 0; border-radius: 4px; display: none; }
        .success { background-color: #dff0d8; color: #3c763d; border: 1px solid #d6e9c6; }
        .error { background-color: #f2dede; color: #a94442; border: 1px solid #ebccd1; }

        /* Loading overlay */
        .loading-overlay { display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 2000; justify-content: center; align-items: center; }
        .loading-spinner { border: 5px solid #f3f3f3; border-top: 5px solid #3498db; border-radius: 50%; width: 50px; height: 50px; animation: spin 1s linear infinite; }
        @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }
    </style>
</head>
<body>
    <h1>Admin Panel: Manage Stock Plans</h1>
    
    <div id="message-container"></div>
    
    <button id="add-stock-btn" class="btn add-btn">Add New Stock Plan</button>
    
    <table id="stocks-table">
        <thead>
            <tr>
                <th>ID</th>
                <th>Name</th>
                <th>Symbol</th>
                <th>Minimum</th>
                <th>Maximum</th>
                <th>Cycle</th>
                <th>Status</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($stocks)): ?>
                <tr>
                    <td colspan="8" style="text-align: center;">No stocks found</td>
                </tr>
            <?php else: ?>
                <?php foreach ($stocks as $stock): ?>
                <tr data-id="<?= $stock['id'] ?>">
                    <td><?= htmlspecialchars($stock['id']) ?></td>
                    <td><?= htmlspecialchars($stock['name']) ?></td>
                    <td><?= htmlspecialchars($stock['symbol']) ?></td>
                    <td><?= number_format($stock['minimum_amount'], 2) ?> USDT</td>
                    <td><?= number_format($stock['maximum_amount'], 2) ?> USDT</td>
                    <td><?= htmlspecialchars($stock['cycle']) ?></td>
                    <td><?= htmlspecialchars($stock['status']) ?></td>
                    <td>
                        <button class="btn edit-btn" data-id="<?= $stock['id'] ?>">Edit</button>
                        <button class="btn delete-btn" data-id="<?= $stock['id'] ?>">Delete</button>
                    </td>
                </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>

    <!-- Modal for Create/Edit -->
    <div id="stock-modal" class="modal">
        <div class="modal-content">
            <span class="close">&times;</span>
            <h2 id="modal-title">Create Stock Plan</h2>
            <form id="stock-form">
                <input type="hidden" id="stock-id" name="id" value="">
                <input type="hidden" id="action" name="action" value="create">
                
                <div class="form-group">
                    <label for="name">Stock Name:</label>
                    <input type="text" id="name" name="name" required placeholder="Stock Name">
                </div>
                
                <div class="form-group">
                    <label for="symbol">Symbol:</label>
                    <input type="text" id="symbol" name="symbol" required placeholder="Symbol (e.g. BTC)">
                </div>
                
                <div class="form-group">
                    <label for="minimum">Minimum Amount (USDT):</label>
                    <input type="number" id="minimum" step="0.01" min="0" name="minimum" required placeholder="Minimum Amount">
                </div>
                
                <div class="form-group">
                    <label for="maximum">Maximum Amount (USDT):</label>
                    <input type="number" id="maximum" step="0.01" min="0" name="maximum" required placeholder="Maximum Amount">
                </div>
                
                <div class="form-group">
                    <label for="cycle">Cycle:</label>
                    <input type="text" id="cycle" name="cycle" required placeholder="Cycle (e.g. 7 days)">
                </div>
                
                <div class="form-group">
                    <label for="status">Status:</label>
                    <select id="status" name="status" required>
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
                
                <button type="submit" class="btn" id="submit-btn">Create Stock Plan</button>
            </form>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div id="delete-modal" class="modal">
        <div class="modal-content">
            <span class="close">&times;</span>
            <h2>Confirm Delete</h2>
            <p>Are you sure you want to delete this stock plan?</p>
            <div>
                <button id="confirm-delete" class="btn delete-btn">Delete</button>
                <button id="cancel-delete" class="btn">Cancel</button>
            </div>
        </div>
    </div>

    <!-- Loading Overlay -->
    <div class="loading-overlay" id="loading-overlay">
        <div class="loading-spinner"></div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Elements
            const stockModal = document.getElementById('stock-modal');
            const deleteModal = document.getElementById('delete-modal');
            const modalTitle = document.getElementById('modal-title');
            const stockForm = document.getElementById('stock-form');
            const addStockBtn = document.getElementById('add-stock-btn');
            const submitBtn = document.getElementById('submit-btn');
            const stocksTable = document.getElementById('stocks-table');
            const closeButtons = document.querySelectorAll('.close');
            const loadingOverlay = document.getElementById('loading-overlay');
            const messageContainer = document.getElementById('message-container');
            let currentDeleteId = null;

            // Show loading
            function showLoading() {
                loadingOverlay.style.display = 'flex';
            }

            // Hide loading
            function hideLoading() {
                loadingOverlay.style.display = 'none';
            }

            // Show message
            function showMessage(message, type) {
                messageContainer.textContent = message;
                messageContainer.className = type;
                messageContainer.style.display = 'block';
                setTimeout(() => {
                    messageContainer.style.display = 'none';
                }, 5000);
            }

            // Reset form
            function resetForm() {
                stockForm.reset();
                document.getElementById('stock-id').value = '';
                document.getElementById('action').value = 'create';
                modalTitle.textContent = 'Create Stock Plan';
                submitBtn.textContent = 'Create Stock Plan';
            }

            // Open modal for adding new stock
            addStockBtn.addEventListener('click', function() {
                resetForm();
                stockModal.style.display = 'block';
            });

            // Handle edit button click
            stocksTable.addEventListener('click', function(e) {
                if (e.target.classList.contains('edit-btn')) {
                    const id = e.target.getAttribute('data-id');
                    showLoading();
                    
                    // Fetch stock data
                    fetch('admin_stocks.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: new URLSearchParams({
                            'action': 'get',
                            'id': id
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        hideLoading();
                        if (data.success) {
                            document.getElementById('stock-id').value = data.data.id;
                            document.getElementById('action').value = 'update';
                            document.getElementById('name').value = data.data.name;
                            document.getElementById('symbol').value = data.data.symbol;
                            document.getElementById('minimum').value = data.data.minimum_amount;
                            document.getElementById('maximum').value = data.data.maximum_amount;
                            document.getElementById('cycle').value = data.data.cycle;
                            document.getElementById('status').value = data.data.status;
                            
                            modalTitle.textContent = 'Edit Stock Plan';
                            submitBtn.textContent = 'Update Stock Plan';
                            stockModal.style.display = 'block';
                        } else {
                            showMessage(data.message || 'Error fetching stock data', 'error');
                        }
                    })
                    .catch(error => {
                        hideLoading();
                        showMessage('Error: ' + error.message, 'error');
                    });
                } else if (e.target.classList.contains('delete-btn')) {
                    const id = e.target.getAttribute('data-id');
                    currentDeleteId = id;
                    deleteModal.style.display = 'block';
                }
            });

            // Handle form submission
            stockForm.addEventListener('submit', function(e) {
                e.preventDefault();
                showLoading();
                
                const formData = new FormData(stockForm);
                
                fetch('admin_stocks.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    hideLoading();
                    if (data.success) {
                        stockModal.style.display = 'none';
                        showMessage(data.message, 'success');
                        // Reload the page to see changes
                        window.location.reload();
                    } else {
                        showMessage(data.message || 'Error processing request', 'error');
                    }
                })
                .catch(error => {
                    hideLoading();
                    showMessage('Error: ' + error.message, 'error');
                });
            });

            // Handle delete confirmation
            document.getElementById('confirm-delete').addEventListener('click', function() {
                if (currentDeleteId) {
                    showLoading();
                    
                    fetch('admin_stocks.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: new URLSearchParams({
                            'action': 'delete',
                            'id': currentDeleteId
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        hideLoading();
                        deleteModal.style.display = 'none';
                        if (data.success) {
                            showMessage(data.message, 'success');
                            // Remove the row from table
                            const row = document.querySelector(`tr[data-id="${currentDeleteId}"]`);
                            if (row) row.remove();
                            
                            // If no more rows, add a "No stocks found" message
                            const tbody = stocksTable.querySelector('tbody');
                            if (tbody.children.length === 0) {
                                const noDataRow = document.createElement('tr');
                                noDataRow.innerHTML = '<td colspan="8" style="text-align: center;">No stocks found</td>';
                                tbody.appendChild(noDataRow);
                            }
                        } else {
                            showMessage(data.message || 'Error deleting stock', 'error');
                        }
                    })
                    .catch(error => {
                        hideLoading();
                        deleteModal.style.display = 'none';
                        showMessage('Error: ' + error.message, 'error');
                    });
                }
            });

            // Handle cancel delete
            document.getElementById('cancel-delete').addEventListener('click', function() {
                deleteModal.style.display = 'none';
                currentDeleteId = null;
            });

            // Close modals when clicking the X
            closeButtons.forEach(function(closeBtn) {
                closeBtn.addEventListener('click', function() {
                    stockModal.style.display = 'none';
                    deleteModal.style.display = 'none';
                    currentDeleteId = null;
                });
            });

            // Close modals when clicking outside
            window.addEventListener('click', function(e) {
                if (e.target === stockModal) {
                    stockModal.style.display = 'none';
                }
                if (e.target === deleteModal) {
                    deleteModal.style.display = 'none';
                    currentDeleteId = null;
                }
            });
        });
    </script>
</body>
</html>