<?php
session_start();
include($_SERVER['DOCUMENT_ROOT'] . '/config.php');

// Verify admin is logged in
if (!isset($_SESSION['admin_id']) || empty($_SESSION['admin_id'])) {
    header('Location: admin_login.php');
    exit();
}
// Enable error logging but disable display
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', dirname(__DIR__) . '/logs/error.log');
// Debug line to check admin session
error_log('Admin ID: ' . $_SESSION['admin_id']);

// Handle user actions (delete, edit, login as, toggle codes)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Debug what action we're processing
    error_log('POST request received: ' . print_r($_POST, true));
    
    if (isset($_POST['user_id'], $_POST['action'])) {
        $user_id = $_POST['user_id'];
        
        // Handle delete action
        if ($_POST['action'] === 'delete') {
            try {
                // Delete the user from the users table
                $stmt = $conn->prepare("DELETE FROM users WHERE id = ?");
                $result = $stmt->execute([$user_id]);
                
                if ($result) {
                    // Log success
                    error_log("User $user_id deleted successfully");
                    // Set success message in session
                    $_SESSION['message'] = "User deleted successfully.";
                } else {
                    // Log error
                    error_log("Failed to delete user $user_id");
                    $_SESSION['error'] = "Failed to delete user.";
                }
                
                // Redirect after deletion - use full URL to avoid path issues
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            } catch (PDOException $e) {
                error_log("Database error: " . $e->getMessage());
                $_SESSION['error'] = "Database error occurred.";
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            }
        }
        
        // Handle login as user action
        if ($_POST['action'] === 'login_as') {
            // Store admin's session data for later restoration
            $_SESSION['admin_original_id'] = $_SESSION['admin_id'];
            
            // Set session as the target user
            $_SESSION['user_id'] = $user_id;
            
            // Fetch user data to set in session
            $stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $user = $stmt->fetch();
            
            $_SESSION['username'] = $user['username'];
            // Unset admin session temporarily
            unset($_SESSION['admin_id']);
            
            // Redirect to user dashboard
            header('Location: /dashboard.php');
            exit;
        }

        // Handle toggle withdraw code
        if ($_POST['action'] === 'toggle_withdraw_code') {
            $status = $_POST['status']; // 'enable' or 'disable'
            $code_value = ($status === 'enable') ? 'required' : 'no_code';
            
            try {
                $stmt = $conn->prepare("UPDATE users SET withdrawal_code = ? WHERE id = ?");
                $result = $stmt->execute([$code_value, $user_id]);
                
                if ($result) {
                    $_SESSION['message'] = "Withdraw code status updated successfully.";
                } else {
                    $_SESSION['error'] = "Failed to update withdraw code status.";
                }
                
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            } catch (PDOException $e) {
                error_log("Database error: " . $e->getMessage());
                $_SESSION['error'] = "Database error occurred.";
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            }
        }

        // Handle toggle login code
       if ($_POST['action'] === 'toggle_login_code') {
    try {
        $stmt = $conn->prepare("SELECT login_code_enabled FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $current = $stmt->fetchColumn();
        
        $new_status = $current ? 0 : 1;
        $update = $conn->prepare("UPDATE users SET login_code_enabled = ? WHERE id = ?");
        $result = $update->execute([$new_status, $user_id]);
        
        if ($result) {
            $_SESSION['message'] = "Login code requirement " . ($new_status ? "enabled" : "disabled");
        } else {
            $_SESSION['error'] = "Failed to update login code status.";
        }
    } catch (PDOException $e) {
        error_log("Database error: " . $e->getMessage());
        $_SESSION['error'] = "Database error occurred.";
    }
    header('Location: ' . $_SERVER['PHP_SELF']);
    exit;
}

        // Handle suspend user action
        if ($_POST['action'] === 'suspend_user') {
            $status = $_POST['status']; // 'suspend' or 'activate'
            $new_status = ($status === 'suspend') ? 'suspended' : 'active';
            
            try {
                $stmt = $conn->prepare("UPDATE users SET status = ? WHERE id = ?");
                $result = $stmt->execute([$new_status, $user_id]);
                
                if ($result) {
                    $_SESSION['message'] = "User status updated successfully.";
                } else {
                    $_SESSION['error'] = "Failed to update user status.";
                }
                
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            } catch (PDOException $e) {
                error_log("Database error: " . $e->getMessage());
                $_SESSION['error'] = "Database error occurred.";
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            }
        }
// Fetch users with their KYC status
$query = "
    SELECT u.*, k.status as kyc_status 
    FROM users u
    LEFT JOIN kyc_requests k ON u.id = k.user_id
    GROUP BY u.id
";

$result = $conn->query($query);
$users = [];
if ($result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        $users[] = $row;
    }
}
        // Handle credit balance
        if ($_POST['action'] === 'credit_balance') {
            $amount = $_POST['amount'];
            
            try {
                $stmt = $conn->prepare("UPDATE users SET balance = balance + ? WHERE id = ?");
                $result = $stmt->execute([$amount, $user_id]);
                
                if ($result) {
                    // Insert a transaction record
                    $stmt = $conn->prepare("INSERT INTO transactions (user_id, type, amount, description, status) VALUES (?, 'credit', ?, 'Admin credit', 'completed')");
                    $stmt->execute([$user_id, $amount]);
                    
                    $_SESSION['message'] = "User balance credited successfully.";
                } else {
                    $_SESSION['error'] = "Failed to credit user balance.";
                }
                
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            } catch (PDOException $e) {
                error_log("Database error: " . $e->getMessage());
                $_SESSION['error'] = "Database error occurred.";
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            }
        }

        // Handle credit earnings
        if ($_POST['action'] === 'credit_earnings') {
            $amount = $_POST['amount'];
            
            try {
                $stmt = $conn->prepare("UPDATE users SET earnings = earnings + ? WHERE id = ?");
                $result = $stmt->execute([$amount, $user_id]);
                
                if ($result) {
                    $_SESSION['message'] = "User earnings credited successfully.";
                } else {
                    $_SESSION['error'] = "Failed to credit user earnings.";
                }
                
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            } catch (PDOException $e) {
                error_log("Database error: " . $e->getMessage());
                $_SESSION['error'] = "Database error occurred.";
                header('Location: ' . $_SERVER['PHP_SELF']);
                exit;
            }
        }
    }
    
    //Fetch all users from database
try {
    $stmt = $conn->query("
        SELECT 
            id, email, password, balance, profit_balance, investment_balance, 
            usdt_balance, is_verified, withdrawal_code, login_code, status,
            withdrawal_code_enabled, login_code_enabled
        FROM users
        ORDER BY id DESC
    ");
    
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    $error = "Database error: " . $e->getMessage();
}
// Handle form actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $user_id = filter_input(INPUT_POST, 'user_id', FILTER_VALIDATE_INT);
    $action = filter_input(INPUT_POST, 'action', FILTER_SANITIZE_STRING);
    
    try {
        switch ($action) {
            case 'toggle_withdrawal_code':
                // Toggle withdrawal code requirement
                $stmt = $conn->prepare("SELECT withdrawal_code_enabled FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $current = $stmt->fetchColumn();
                
                $new_status = $current ? 0 : 1;
                $update = $conn->prepare("UPDATE users SET withdrawal_code_enabled = ? WHERE id = ?");
                $update->execute([$new_status, $user_id]);
                
                $_SESSION['message'] = "Withdrawal code requirement " . ($new_status ? "enabled" : "disabled");
                break;
                
            case 'toggle_login_code':
                // Toggle login code requirement
                $stmt = $conn->prepare("SELECT login_code_enabled FROM users WHERE id = ?");
                $stmt->execute([$user_id]);
                $current = $stmt->fetchColumn();
                
                $new_status = $current ? 0 : 1;
                $update = $conn->prepare("UPDATE users SET login_code_enabled = ? WHERE id = ?");
                $update->execute([$new_status, $user_id]);
                
                $_SESSION['message'] = "Login code requirement " . ($new_status ? "enabled" : "disabled");
                break;
                
            // ... (keep your other action cases)
                
            default:
                throw new Exception("Invalid action");
        }
        
        header("Location: ".$_SERVER['PHP_SELF']);
        exit();
        
    } catch (Exception $e) {
        $_SESSION['error'] = "Error: " . $e->getMessage();
        header("Location: ".$_SERVER['PHP_SELF']);
        exit();
    }
}
    
    // Handle edit form submission
    if (isset($_POST['edit_user_id'])) {
        $user_id = $_POST['edit_user_id'];
        $username = $_POST['username'];
        $email = $_POST['email'];
        $phone = $_POST['phone'];
        $password = isset($_POST['password']) && !empty($_POST['password']) ? password_hash($_POST['password'], PASSWORD_DEFAULT) : null;
        
        try {
            if ($password) {
                $stmt = $conn->prepare("UPDATE users SET username = ?, email = ?, phone = ?, password = ? WHERE id = ?");
                $result = $stmt->execute([$username, $email, $phone, $password, $user_id]);
            } else {
                $stmt = $conn->prepare("UPDATE users SET username = ?, email = ?, phone = ? WHERE id = ?");
                $result = $stmt->execute([$username, $email, $phone, $user_id]);
            }
            
            if ($result) {
                $_SESSION['message'] = "User details updated successfully.";
            } else {
                $_SESSION['error'] = "Failed to update user details.";
            }
            
            // Redirect after update
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        } catch (PDOException $e) {
            error_log("Database error: " . $e->getMessage());
            $_SESSION['error'] = "Database error occurred.";
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        }
    }
}

// Fetch all users with detailed information
// Fetch all users with detailed information
$stmt = $conn->prepare("
    SELECT u.*, 
           COALESCE(SUM(CASE WHEN t.transaction_type = 'profit' THEN t.amount ELSE 0 END), 0) as total_profit
    FROM users u
    LEFT JOIN transactions t ON u.id = t.user_id
    GROUP BY u.id
    ORDER BY u.username ASC
");
$stmt->execute();
$users = $stmt->fetchAll();
// Handle edit modal data fetching
$edit_user = null;
if (isset($_GET['edit']) && is_numeric($_GET['edit'])) {
    $edit_id = $_GET['edit'];
    $stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$edit_id]);
    $edit_user = $stmt->fetch();
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
        }

        :root {
            --sidebar-bg: #14213d;
            --sidebar-text: #a8b2d1;
            --sidebar-hover: #1a2c5b;
            --active-link: #2a4494;
            --primary-text: #333;
            --secondary-text: #777;
            --card-bg: #fff;
            --main-bg: #f0f2f5;
            --deposit-color: #3498db;
            --withdraw-color: #e74c3c;
            --button-color: #0d6efd;
            --hover-color: #0b5ed7;
        }

        body {
            display: flex;
            background-color: var(--main-bg);
            min-height: 100vh;
        }

        /* Mobile nav toggle */
        .mobile-nav-toggle {
            display: none;
            position: fixed;
            top: 15px;
            left: 15px;
            z-index: 1050;
            background-color: var(--sidebar-bg);
            color: var(--sidebar-text);
            width: 40px;
            height: 40px;
            border-radius: 4px;
            justify-content: center;
            align-items: center;
            cursor: pointer;
            border: none;
            font-size: 20px;
        }

        /* Overlay for mobile */
        .overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 900;
        }

        /* Sidebar Styles */
        .sidebar {
            width: 250px;
            background-color: var(--sidebar-bg);
            color: var(--sidebar-text);
            height: 100vh;
            position: fixed;
            transition: all 0.3s ease;
            overflow-y: auto;
            z-index: 1000;
        }

        .sidebar-header {
            padding: 20px;
            font-size: 18px;
            font-weight: bold;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }

        .sidebar-menu {
            margin-top: 20px;
        }

        .section-title {
            padding: 10px 20px;
            font-size: 12px;
            text-transform: uppercase;
            letter-spacing: 1px;
            color: #8a94a7;
            margin-top: 10px;
        }

        .menu-item {
            display: flex;
            align-items: center;
            padding: 12px 20px;
            text-decoration: none;
            color: var(--sidebar-text);
            transition: all 0.3s ease;
        }

        .menu-item:hover {
            background-color: var(--sidebar-hover);
        }

        .menu-item.active {
            background-color: var(--active-link);
            color: white;
        }

        .menu-item i {
            margin-right: 10px;
            width: 20px;
            text-align: center;
        }

        /* Main Content */
        .main-content {
            flex: 1;
            margin-left: 250px;
            padding: 20px;
            width: calc(100% - 250px);
            transition: all 0.3s ease;
        }

        .top-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid #eee;
        }

        .page-icon {
            font-size: 20px;
            color: var(--primary-text);
        }

        .user-info {
            display: flex;
            align-items: center;
        }

        .user-icon {
            width: 32px;
            height: 32px;
            background-color: var(--button-color);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            margin-right: 10px;
        }

        .user-role {
            display: flex;
            flex-direction: column;
        }

        .role-name {
            font-size: 12px;
            color: var(--secondary-text);
        }

        /* Overview Section */
        .overview-section {
            margin-bottom: 30px;
        }

        .overview-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .overview-title {
            font-size: 24px;
            font-weight: 500;
            color: var(--primary-text);
        }

        .overview-description {
            font-size: 14px;
            color: var(--secondary-text);
            margin-top: 5px;
        }

        .view-users-btn {
            padding: 8px 16px;
            background-color: var(--button-color);
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            display: flex;
            align-items: center;
            transition: background-color 0.3s;
        }

        .view-users-btn:hover {
            background-color: var(--hover-color);
        }

        .view-users-btn i {
            margin-right: 6px;
        }

        /* Cards Grid */
        .cards-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }

        .card {
            background-color: var(--card-bg);
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
            padding: 20px;
            position: relative;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }

        .card-title {
            font-size: 16px;
            color: var(--primary-text);
        }

        .card-icon {
            font-size: 14px;
            color: var(--secondary-text);
            cursor: pointer;
        }

        .card-amount {
            font-size: 28px;
            font-weight: 600;
            margin-bottom: 20px;
        }

        .deposit-bar {
            height: 4px;
            width: 100%;
            background-color: var(--deposit-color);
            border-radius: 2px;
        }

        .withdraw-bar {
            height: 4px;
            width: 100%;
            background-color: var(--withdraw-color);
            border-radius: 2px;
        }

        /* Stats Card */
        .stats-card {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 20px;
        }

        .stat-item {
            margin-bottom: 15px;
        }

        .stat-label {
            display: flex;
            align-items: center;
            margin-bottom: 5px;
            font-size: 14px;
            color: var(--secondary-text);
        }

        .stat-label-circle {
            width: 12px;
            height: 12px;
            border-radius: 50%;
            margin-right: 8px;
        }

        .deposit-circle {
            background-color: #f39c12;
        }

        .withdraw-circle {
            background-color: #e74c3c;
        }

        .transaction-circle {
            background-color: #9b59b6;
        }

        .users-circle {
            background-color: #3498db;
        }

        .stat-value {
            font-size: 24px;
            font-weight: 500;
        }

        /* Footer */
        .footer {
            font-size: 12px;
            color: var(--secondary-text);
            margin-top: 40px;
            padding-top: 10px;
            border-top: 1px solid #eee;
        }

        /* Responsive */
        @media (max-width: 992px) {
            .sidebar {
                width: 80px;
            }
            .sidebar-header {
                padding: 15px 10px;
                text-align: center;
                font-size: 14px;
            }
            .menu-item span, .section-title {
                display: none;
            }
            .menu-item {
                justify-content: center;
                padding: 15px 0;
            }
            .menu-item i {
                margin-right: 0;
                font-size: 18px;
            }
            .main-content {
                margin-left: 80px;
                width: calc(100% - 80px);
            }
        }

        @media (max-width: 768px) {
            .cards-grid {
                grid-template-columns: 1fr;
            }
            .stats-card {
                grid-template-columns: 1fr;
            }
            .mobile-nav-toggle {
                display: flex;
            }
            .sidebar {
                transform: translateX(-100%);
                width: 250px;
            }
            .sidebar.active {
                transform: translateX(0);
            }
            .menu-item span, .section-title {
                display: block;
            }
            .menu-item {
                justify-content: flex-start;
                padding: 12px 20px;
            }
            .menu-item i {
                margin-right: 10px;
            }
            .main-content {
                margin-left: 0;
                width: 100%;
                padding-top: 60px;
            }
            .overlay.active {
                display: block;
            }
        }

        @media (max-width: 576px) {
            .overview-header {
                flex-direction: column;
                align-items: flex-start;
            }
            .view-users-btn {
                margin-top: 10px;
            }
        }
    </style>
</head>
<body>
    <!-- Mobile menu toggle -->
    <button class="mobile-nav-toggle" id="navToggle">
        <i class="fas fa-bars"></i>
    </button>

             <!-- Overlay for closing the sidebar on mobile -->
    <div class="overlay" id="overlay"></div>

    <!-- Sidebar -->
    <div class="sidebar" id="sidebar">
        <div class="sidebar-header">ADMIN DASHBOARD</div>
        <div class="sidebar-menu">
            <a href="/admin/admin_dashboard.php" class="menu-item active">
                <i class="fas fa-th-large"></i>
                <span>Dashboard</span>
            </a>
            <a href="/admin/coin-deposit-request.php" class="menu-item">
                <i class="fas fa-arrow-circle-down"></i>
                <span>Deposit Coin</span>
            </a>
             <a href="/admin/deposit_requests.php" class="menu-item">
                <i class="fas fa-money-check-alt"></i>
                <span>Deposit Check</span>
            </a>
            <a href="/admin/transfer-request.php" class="menu-item">
                <i class="fas fa-arrow-circle-up"></i>
                <span>Withdraw</span>
            </a>
            <a href="/admin/manage-payment-method.php" class="menu-item">
                <i class="fas fa-exchange-alt"></i>
                <span>Manage Payment Method</span>
            </a>
            <a href="/admin/admin_staking.php" class="menu-item">
                <i class="fas fa-lock"></i>
                <span>Manage Staking</span>
            </a>
            <a href="/admin/admin_staking_transactions.php" class="menu-item">
                <i class="fas fa-lock"></i>
                <span>Manage Staking Transations</span>
            </a>
            <a href="/admin/admin_stocks.php" class="menu-item">
                <i class="fas fa-chart-bar"></i>
                <span>Manage Stocks</span>
            </a>
            <a href="/admin/admin_stock_transactions.php" class="menu-item">
                <i class="fas fa-chart-bar"></i>
                <span>Manage Stocks Transations</span>
            </a>
            <a href="/admin/admin_bots.php" class="menu-item">
                <i class="fas fa-robot"></i>
                <span>Manage Bot</span>
            </a>
            <a href="/admin/investment.php" class="menu-item">
                <i class="fas fa-coins"></i>
                <span>Manage Investment</span>
            </a>
            
            <a href="/admin/crypto_orders.php" class="menu-item">
                <i class="fas fa-coins"></i>
                <span>Manage Crypto Order</span>
            </a>
            
            <a href="/admin/update_balance.php" class="menu-item">
                <i class="fas fa-file-invoice-dollar"></i>
                <span>Update Balances</span>
            </a>
            <a href="/admin/kyc-request.php" class="menu-item">
                <i class="fas fa-id-card"></i>
                <span>KYC</span>
            </a>
            <a href="/admin/admin_send_notification.php" class="menu-item">
                <i class="fas fa-bell"></i>
                <span>Manage Notifications</span>
            </a>

            <div class="section-title">USER MANAGEMENT</div>
            <a href="/admin/manage_users.php" class="menu-item">
                <i class="fas fa-users"></i>
                <span>Manage Users</span>
            </a>
            <div class="section-title">Approve SignUp</div>
            <a href="/admin/admin_approval.php" class="menu-item">
                <i class="fas fa-user-check"></i>
                <span>Approve SignUp</span>
            </a>
            <div class="section-title">Change Password</div>
            <a href="/admin/change_password.php" class="menu-item">
                <i class="fas fa-unlock-alt"></i>
                <span>Change Password</span>
            </a>

            <div class="section-title">MANAGEMENT</div>
            <a href="/admin/logout.php" class="menu-item">
                <i class="fas fa-sign-out"></i>
                <span>Sign Out</span>
            </a>
        </div>
    </div>


    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="top-bar">
            <div class="page-icon">
                <i class="fas fa-clipboard"></i>
            </div>
            <div class="user-info">
                <div class="user-icon">
                    <i class="fas fa-user"></i>
                </div>
                <div class="user-role">
                    <div>Admin</div>
                    <div class="role-name">Administrator <i class="fas fa-chevron-down"></i></div>
                </div>
            </div>
        </div>

        <!-- Overview Section -->
        <div class="overview-section">
            <div class="overview-header">
                <div>
                    <h2 class="overview-title">Overview</h2>
                    <p class="overview-description">Here is an insight of what's going on.</p>
                </div>
                <button class="view-users-btn">
                    <i class="fas fa-users"></i> View All Users
                </button>
            </div>

    
<style>
/* Container styling */
.container {
    background-color: var(--card-bg);
    border-radius: 8px;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
    padding: 20px;
    overflow-x: auto;
    margin-bottom: 30px;
}

.container h1 {
    font-size: 20px;
    color: var(--primary-text);
    margin-bottom: 20px;
}

/* Messages styling */
.message {
    padding: 12px 15px;
    margin-bottom: 20px;
    border-radius: 5px;
    font-size: 14px;
}

.success {
    background-color: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

.error {
    background-color: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

/* Table styling */
table {
    width: 100%;
    border-collapse: collapse;
    margin-bottom: 20px;
    font-size: 14px;
}

table th {
    background-color: #f8f9fa;
    color: var(--primary-text);
    font-weight: 600;
    text-align: left;
    padding: 12px 15px;
    border: 1px solid #e9ecef;
}

table td {
    padding: 12px 15px;
    border: 1px solid #e9ecef;
    vertical-align: middle;
}

/* Status indicators */
.verified {
    color: #28a745;
    font-weight: 500;
}

.not-verified {
    color: #dc3545;
    font-weight: 500;
}

.code-status {
    display: inline-block;
    padding: 4px 8px;
    border-radius: 4px;
    font-size: 12px;
    font-weight: 500;
}

.code-required {
    background-color: #fff3cd;
    color: #856404;
}

.code-no {
    background-color: #d4edda;
    color: #155724;
}

/* Toggle switch styling */
.toggle-switch {
    position: relative;
    display: inline-block;
    width: 48px;
    height: 24px;
    vertical-align: middle;
}

.toggle-switch input {
    opacity: 0;
    width: 0;
    height: 0;
}

.slider {
    position: absolute;
    cursor: pointer;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background-color: #ccc;
    transition: .4s;
    border-radius: 24px;
}

.slider:before {
    position: absolute;
    content: "";
    height: 18px;
    width: 18px;
    left: 3px;
    bottom: 3px;
    background-color: white;
    transition: .4s;
    border-radius: 50%;
}

input:checked + .slider {
    background-color: var(--button-color);
}

input:focus + .slider {
    box-shadow: 0 0 1px var(--button-color);
}

input:checked + .slider:before {
    transform: translateX(24px);
}

.toggle-label {
    margin-left: 8px;
    font-size: 13px;
    color: var(--secondary-text);
}

/* Action dropdown styling */
.action-dropdown {
    position: relative;
    display: inline-block;
}

.action-btn {
    background-color: transparent;
    border: none;
    font-size: 18px;
    cursor: pointer;
    color: var(--secondary-text);
    padding: 5px 10px;
}

.dropdown-content {
    display: none;
    position: absolute;
    right: 0;
    background-color: white;
    min-width: 160px;
    box-shadow: 0px 8px 16px 0px rgba(0,0,0,0.2);
    z-index: 1;
    border-radius: 4px;
}

.dropdown-content a {
    color: var(--primary-text);
    padding: 10px 12px;
    text-decoration: none;
    display: block;
    font-size: 13px;
    border-bottom: 1px solid #f5f5f5;
}

.dropdown-content a:last-child {
    border-bottom: none;
}

.dropdown-content a:hover {
    background-color: #f5f5f5;
}

.action-dropdown:hover .dropdown-content {
    display: block;
}

/* Modal styling */
.modal {
    display: none;
    position: fixed;
    z-index: 1100;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    overflow: auto;
    background-color: rgba(0,0,0,0.5);
}

.modal-content {
    background-color: var(--card-bg);
    margin: 10% auto;
    padding: 20px;
    border-radius: 8px;
    width: 80%;
    max-width: 500px;
    box-shadow: 0 5px 15px rgba(0,0,0,0.2);
    position: relative;
}

.close {
    position: absolute;
    right: 20px;
    top: 15px;
    color: var(--secondary-text);
    font-size: 28px;
    font-weight: bold;
    cursor: pointer;
}

.close:hover {
    color: var(--primary-text);
}

.form-group {
    margin-bottom: 15px;
}

.form-group label {
    display: block;
    margin-bottom: 5px;
    font-weight: 500;
    color: var(--primary-text);
}

.form-group input {
    width: 100%;
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
}

.form-actions {
    display: flex;
    justify-content: flex-end;
    margin-top: 20px;
    gap: 10px;
}

.btn {
    padding: 8px 16px;
    border-radius: 4px;
    border: none;
    cursor: pointer;
    font-size: 14px;
}

.btn-primary {
    background-color: var(--button-color);
    color: white;
}

.btn-secondary {
    background-color: #6c757d;
    color: white;
}

/* Responsive */
@media (max-width: 992px) {
    .container {
        padding: 15px;
    }
}

@media (max-width: 768px) {
    table {
        font-size: 13px;
    }
    
    table th, table td {
        padding: 8px 10px;
    }
    
    .container h1 {
        font-size: 18px;
    }
}

@media (max-width: 576px) {
    .container {
        padding: 10px;
    }
    
    table {
        display: block;
        overflow-x: auto;
        white-space: nowrap;
    }
    
    .modal-content {
        width: 95%;
        margin: 5% auto;
    }
}
</style>

<!-- User Management Table -->
<div class="container">
    <h1>User Management</h1>

    <?php if (isset($_SESSION['message'])): ?>
        <div class="message success">
            <?php echo $_SESSION['message']; unset($_SESSION['message']); ?>
        </div>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
        <div class="message error">
            <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
        </div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="message error"><?php echo $error; ?></div>
    <?php endif; ?>
    
    <table>
        <thead>
            <tr>
                <th>Email</th>
                <th>Password</th>
                <th>Balance</th>
                <th>Profit</th>
                <th>Investment Balance</th>
                <th>Usdt Balance</th>
                <th>Verified</th>
                <th>Withdrawal Code</th>
                <th>Login Code</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($users as $user): ?>
                <tr>
                    <td><?php echo htmlspecialchars($user['email']); ?></td>
                    <td><?php echo htmlspecialchars($user['password']); ?></td>
                    <td><?php echo number_format($user['balance'], 2); ?> USD</td>
                    <td><?php echo number_format($user['profit_balance'], 2); ?> USD</td>
                    <td><?php echo number_format($user['investment_balance'], 2); ?> USD</td>
                    <td><?php echo number_format($user['usdt_balance'], 2); ?> USD</td>
                    <td class="<?php echo $user['is_verified'] ? 'verified' : 'not-verified'; ?>">
                        <?php echo $user['is_verified'] ? 'Verified' : 'Not Verified'; ?>
                    </td>
                    <td>
                        <form method="post" style="display:inline;">
                            <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                            <input type="hidden" name="action" value="toggle_withdrawal_code">
                            <label class="toggle-switch">
                                <input type="checkbox" 
                                       onchange="this.form.submit()" 
                                       <?php echo $user['withdrawal_code_enabled'] ? 'checked' : ''; ?>>
                                <span class="slider"></span>
                            </label>
                            <span class="toggle-label">
                                <?php echo $user['withdrawal_code_enabled'] ? 'Required' : 'Not Required'; ?>
                            </span>
                        </form>
                    </td>
                    <td>
    <form method="post" style="display:inline;">
        <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
        <input type="hidden" name="action" value="toggle_login_code">
        <label class="toggle-switch">
            <input type="checkbox" 
                   onchange="this.form.submit()" 
                   <?php echo $user['login_code_enabled'] ? 'checked' : ''; ?>>
            <span class="slider"></span>
        </label>
        <span class="toggle-label">
            <?php echo $user['login_code_enabled'] ? 'Enabled' : 'Disabled'; ?>
        </span>
    </form>
</td>
                    <td>
                        <div class="action-dropdown">
                            <button class="action-btn">•••</button>
                            <div class="dropdown-content">
                                <a href="/admin/update_balance.php?user_id=<?php echo $user['id']; ?>">+ Credit Balance</a>
                                <a href="?edit=<?php echo $user['id']; ?>">Edit Details</a>
                                <form method="post" style="display:inline;">
                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                    <input type="hidden" name="action" value="login_as">
                                    <a href="#" onclick="this.parentNode.submit();">Login to account</a>
                                </form>
                                <form method="post" style="display:inline;">
                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                    <input type="hidden" name="action" value="suspend_user">
                                    <input type="hidden" name="status" value="<?php echo $user['status'] === 'active' ? 'suspend' : 'activate'; ?>">
                                </form>
                                <form method="post" style="display:inline;">
                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                    <input type="hidden" name="action" value="delete">
                                    <a href="#" onclick="if(confirm('Are you sure you want to delete this user?')) { this.parentNode.submit(); }">Delete User</a>
                                </form>
                            </div>
                        </div>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>

<!-- Edit User Modal -->
<?php if ($edit_user): ?>
<div id="editModal" class="modal" style="display: block;">
    <div class="modal-content">
        <span class="close" onclick="document.getElementById('editModal').style.display='none'">&times;</span>
        <h2>Edit User Details</h2>
        <form method="post">
            <input type="hidden" name="edit_user_id" value="<?php echo $edit_user['id']; ?>">
            
            <div class="form-group">
                <label for="username">Username:</label>
                <input type="text" id="username" name="username" value="<?php echo htmlspecialchars($edit_user['username']); ?>" required>
            </div>
            
            <div class="form-group">
                <label for="email">Email:</label>
                <input type="email" id="email" name="email" value="<?php echo htmlspecialchars($edit_user['email']); ?>" required>
            </div>
            
            <div class="form-group">
                <label for="phone">Phone:</label>
                <input type="text" id="phone" name="phone" value="<?php echo htmlspecialchars($edit_user['phone']); ?>">
            </div>
            
            <div class="form-group">
                <label for="password">New Password (leave blank to keep current):</label>
                <input type="password" id="password" name="password">
            </div>
            
            <div class="form-actions">
                <button type="button" class="btn btn-secondary" onclick="document.getElementById('editModal').style.display='none'">Cancel</button>
                <button type="submit" class="btn btn-primary">Save Changes</button>
            </div>
        </form>
    </div>
</div>
<?php endif; ?>

    <script>
        // Close modal when clicking outside of it
        window.onclick = function(event) {
            var modal = document.getElementById('editModal');
            if (event.target == modal) {
                modal.style.display = "none";
            }
        }
        // Get references to elements
        const navToggle = document.getElementById('navToggle');
        const sidebar = document.getElementById('sidebar');
        const overlay = document.getElementById('overlay');
        const mainContent = document.getElementById('mainContent');

        // Toggle sidebar
        navToggle.addEventListener('click', function() {
            sidebar.classList.toggle('active');
            overlay.classList.toggle('active');
        });

        // Close sidebar when clicking on overlay
        overlay.addEventListener('click', function() {
            sidebar.classList.remove('active');
            overlay.classList.remove('active');
        });

        // Close sidebar when clicking on main content (for mobile)
        mainContent.addEventListener('click', function() {
            if (window.innerWidth <= 768 && sidebar.classList.contains('active')) {
                sidebar.classList.remove('active');
                overlay.classList.remove('active');
            }
        });

        // Handle window resize
        window.addEventListener('resize', function() {
            if (window.innerWidth > 768) {
                sidebar.classList.remove('active');
                overlay.classList.remove('active');
            }
        });

        // Close sidebar when clicking menu items (for mobile)
        const menuItems = document.querySelectorAll('.menu-item');
        menuItems.forEach(item => {
            item.addEventListener('click', function() {
                if (window.innerWidth <= 768) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                }
            });
        });
    </script>
</body>
</html>