<?php
session_start();
require_once($_SERVER['DOCUMENT_ROOT'] . '/config.php');


// Verify admin is logged in
if (!isset($_SESSION['admin_id']) || empty($_SESSION['admin_id'])) {
    header('Location: admin_login.php');
    exit();
}
// Handle JSON requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_SERVER['CONTENT_TYPE']) && strpos($_SERVER['CONTENT_TYPE'], 'application/json') !== false) {
    header('Content-Type: application/json');
    
    try {
        $input = file_get_contents('php://input');
        if (empty($input)) {
            throw new Exception('Empty request body');
        }
        
        $data = json_decode($input, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Invalid JSON: ' . json_last_error_msg());
        }
        
        if (!isset($data['action'], $data['transaction_id'])) {
            throw new Exception('Invalid request parameters');
        }
        
        $conn->beginTransaction();
        
        $txId = (int)$data['transaction_id'];
        $action = $data['action'];
        $adminId = $_SESSION['user_id'];
        
        // Get transaction with FOR UPDATE lock
        $txStmt = $conn->prepare("SELECT * FROM bot_transactions WHERE id = ? FOR UPDATE");
        if (!$txStmt->execute([$txId])) {
            throw new Exception('Failed to fetch transaction');
        }
        $tx = $txStmt->fetch();
        
        if (!$tx) {
            throw new Exception('Transaction not found');
        }
        
        if ($tx['status'] !== 'PENDING') {
            throw new Exception('Transaction already processed');
        }
        
        if ($action === 'approve') {
            $updateStmt = $conn->prepare("
                UPDATE bot_transactions 
                SET status = 'COMPLETED', approved_at = NOW(), admin_id = ?
                WHERE id = ?
            ");
            if (!$updateStmt->execute([$adminId, $txId])) {
                throw new Exception('Failed to approve transaction');
            }
        } 
        elseif ($action === 'reject') {
            // Refund user
            $refundStmt = $conn->prepare("UPDATE users SET balance = balance + ? WHERE id = ?");
            if (!$refundStmt->execute([$tx['amount'], $tx['user_id']])) {
                throw new Exception('Failed to refund user');
            }
                
            $updateStmt = $conn->prepare("
                UPDATE bot_transactions 
                SET status = 'REJECTED', admin_id = ?
                WHERE id = ?
            ");
            if (!$updateStmt->execute([$adminId, $txId])) {
                throw new Exception('Failed to reject transaction');
            }
        }
        else {
            throw new Exception('Invalid action');
        }
        
        $conn->commit();
        echo json_encode(['success' => true]);
        
    } catch (Exception $e) {
        if ($conn->inTransaction()) {
            $conn->rollBack();
        }
        http_response_code(400);
        echo json_encode(['error' => $e->getMessage()]);
    }
    exit;
}

// Handle bot management (create/edit/delete)
// Handle bot management (create/edit/delete)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['bot_action'])) {
    try {
        $action = $_POST['bot_action'];
        $conn->beginTransaction();
        
        if ($action === 'create') {
            // Validate and create new bot
            $required = ['name', 'type', 'description', 'symbol', 'minimum_amount', 'maximum_amount', 'return_rate', 'cycle'];
            foreach ($required as $field) {
                if (empty($_POST[$field])) {
                    throw new Exception("$field is required");
                }
            }

            $stmt = $conn->prepare("
                INSERT INTO trading_bots 
                (name, type, description, symbol, minimum_amount, maximum_amount, return_rate, cycle, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            $stmt->execute([
                $_POST['name'],
                $_POST['type'],
                $_POST['description'],
                $_POST['symbol'],
                $_POST['minimum_amount'],
                $_POST['maximum_amount'],
                $_POST['return_rate'],
                $_POST['cycle']
            ]);
            
            $_SESSION['success_message'] = 'Bot created successfully';
        }
        elseif ($action === 'update') {
            // Validate and update bot
            if (empty($_POST['id'])) {
                throw new Exception("Bot ID is required for update");
            }
            
            $required = ['name', 'type', 'description', 'symbol', 'minimum_amount', 'maximum_amount', 'return_rate', 'cycle'];
            foreach ($required as $field) {
                if (empty($_POST[$field])) {
                    throw new Exception("$field is required");
                }
            }

            $stmt = $conn->prepare("
                UPDATE trading_bots SET
                name = ?,
                type = ?,
                description = ?,
                symbol = ?,
                minimum_amount = ?,
                maximum_amount = ?,
                return_rate = ?,
                cycle = ?
                WHERE id = ?
            ");
            $result = $stmt->execute([
                $_POST['name'],
                $_POST['type'],
                $_POST['description'],
                $_POST['symbol'],
                $_POST['minimum_amount'],
                $_POST['maximum_amount'],
                $_POST['return_rate'],
                $_POST['cycle'],
                $_POST['id']
            ]);
            
            if (!$result) {
                throw new Exception("Failed to update bot");
            }
            
            $_SESSION['success_message'] = 'Bot updated successfully';
        }
        elseif ($action === 'delete') {
            // Delete bot (existing code remains the same)
            if (empty($_POST['id'])) {
                throw new Exception("Bot ID is required");
            }
            
            // Check if bot is in use
            $inUse = $conn->prepare("SELECT COUNT(*) FROM bot_transactions WHERE bot_id = ?");
            $inUse->execute([$_POST['id']]);
            if ($inUse->fetchColumn() > 0) {
                throw new Exception("Cannot delete bot - it has active transactions");
            }
            
            $stmt = $conn->prepare("DELETE FROM trading_bots WHERE id = ?");
            $stmt->execute([$_POST['id']]);
            
            $_SESSION['success_message'] = 'Bot deleted successfully';
        }
        
        $conn->commit();
        header("Location: admin_bots.php");
        exit;
        
    } catch (Exception $e) {
        $conn->rollBack();
        $_SESSION['error_message'] = $e->getMessage();
        header("Location: admin_bots.php");
        exit;
    }
}
// Load data
$pendingBots = $conn->query("
    SELECT t.*, u.username 
    FROM bot_transactions t
    JOIN users u ON t.user_id = u.id
    WHERE t.status = 'PENDING'
    ORDER BY t.created_at DESC
")->fetchAll();

$allBots = $conn->query("SELECT * FROM trading_bots ORDER BY created_at DESC")->fetchAll();

// Get bot for editing if ID is provided
$editBot = null;
if (isset($_GET['edit'])) {
    $stmt = $conn->prepare("SELECT * FROM trading_bots WHERE id = ?");
    $stmt->execute([$_GET['edit']]);
    $editBot = $stmt->fetch();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, 'Open Sans', 'Helvetica Neue', sans-serif;
        }

        :root {
            --sidebar-bg: #14213d;
            --sidebar-text: #a8b2d1;
            --sidebar-hover: #1a2c5b;
            --active-link: #2a4494;
            --primary-text: #333;
            --secondary-text: #777;
            --card-bg: #fff;
            --main-bg: #f0f2f5;
            --deposit-color: #3498db;
            --withdraw-color: #e74c3c;
            --button-color: #0d6efd;
            --hover-color: #0b5ed7;
        }

        body {
            display: flex;
            background-color: var(--main-bg);
            min-height: 100vh;
        }

        /* Mobile nav toggle */
        .mobile-nav-toggle {
            display: none;
            position: fixed;
            top: 15px;
            left: 15px;
            z-index: 1050;
            background-color: var(--sidebar-bg);
            color: var(--sidebar-text);
            width: 40px;
            height: 40px;
            border-radius: 4px;
            justify-content: center;
            align-items: center;
            cursor: pointer;
            border: none;
            font-size: 20px;
        }

        /* Overlay for mobile */
        .overlay {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 900;
        }

        /* Sidebar Styles */
        .sidebar {
            width: 250px;
            background-color: var(--sidebar-bg);
            color: var(--sidebar-text);
            height: 100vh;
            position: fixed;
            transition: all 0.3s ease;
            overflow-y: auto;
            z-index: 1000;
        }

        .sidebar-header {
            padding: 20px;
            font-size: 18px;
            font-weight: bold;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }

        .sidebar-menu {
            margin-top: 20px;
        }

        .section-title {
            padding: 10px 20px;
            font-size: 12px;
            text-transform: uppercase;
            letter-spacing: 1px;
            color: #8a94a7;
            margin-top: 10px;
        }

        .menu-item {
            display: flex;
            align-items: center;
            padding: 12px 20px;
            text-decoration: none;
            color: var(--sidebar-text);
            transition: all 0.3s ease;
        }

        .menu-item:hover {
            background-color: var(--sidebar-hover);
        }

        .menu-item.active {
            background-color: var(--active-link);
            color: white;
        }

        .menu-item i {
            margin-right: 10px;
            width: 20px;
            text-align: center;
        }

        /* Main Content */
        .main-content {
            flex: 1;
            margin-left: 250px;
            padding: 20px;
            width: calc(100% - 250px);
            transition: all 0.3s ease;
        }

        .top-bar {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid #eee;
        }

        .page-icon {
            font-size: 20px;
            color: var(--primary-text);
        }

        .user-info {
            display: flex;
            align-items: center;
        }

        .user-icon {
            width: 32px;
            height: 32px;
            background-color: var(--button-color);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            margin-right: 10px;
        }

        .user-role {
            display: flex;
            flex-direction: column;
        }

        .role-name {
            font-size: 12px;
            color: var(--secondary-text);
        }

        /* Overview Section */
        .overview-section {
            margin-bottom: 30px;
        }

        .overview-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .overview-title {
            font-size: 24px;
            font-weight: 500;
            color: var(--primary-text);
        }

        .overview-description {
            font-size: 14px;
            color: var(--secondary-text);
            margin-top: 5px;
        }

        .view-users-btn {
            padding: 8px 16px;
            background-color: var(--button-color);
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
            display: flex;
            align-items: center;
            transition: background-color 0.3s;
        }

        .view-users-btn:hover {
            background-color: var(--hover-color);
        }

        .view-users-btn i {
            margin-right: 6px;
        }

        /* Cards Grid */
        .cards-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }

        .card {
            background-color: var(--card-bg);
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05);
            padding: 20px;
            position: relative;
        }

        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }

        .card-title {
            font-size: 16px;
            color: var(--primary-text);
        }

        .card-icon {
            font-size: 14px;
            color: var(--secondary-text);
            cursor: pointer;
        }

        .card-amount {
            font-size: 28px;
            font-weight: 600;
            margin-bottom: 20px;
        }

        .deposit-bar {
            height: 4px;
            width: 100%;
            background-color: var(--deposit-color);
            border-radius: 2px;
        }

        .withdraw-bar {
            height: 4px;
            width: 100%;
            background-color: var(--withdraw-color);
            border-radius: 2px;
        }

        /* Stats Card */
        .stats-card {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 20px;
        }

        .stat-item {
            margin-bottom: 15px;
        }

        .stat-label {
            display: flex;
            align-items: center;
            margin-bottom: 5px;
            font-size: 14px;
            color: var(--secondary-text);
        }

        .stat-label-circle {
            width: 12px;
            height: 12px;
            border-radius: 50%;
            margin-right: 8px;
        }

        .deposit-circle {
            background-color: #f39c12;
        }

        .withdraw-circle {
            background-color: #e74c3c;
        }

        .transaction-circle {
            background-color: #9b59b6;
        }

        .users-circle {
            background-color: #3498db;
        }

        .stat-value {
            font-size: 24px;
            font-weight: 500;
        }

        /* Footer */
        .footer {
            font-size: 12px;
            color: var(--secondary-text);
            margin-top: 40px;
            padding-top: 10px;
            border-top: 1px solid #eee;
        }

        /* Responsive */
        @media (max-width: 992px) {
            .sidebar {
                width: 80px;
            }
            .sidebar-header {
                padding: 15px 10px;
                text-align: center;
                font-size: 14px;
            }
            .menu-item span, .section-title {
                display: none;
            }
            .menu-item {
                justify-content: center;
                padding: 15px 0;
            }
            .menu-item i {
                margin-right: 0;
                font-size: 18px;
            }
            .main-content {
                margin-left: 80px;
                width: calc(100% - 80px);
            }
        }

        @media (max-width: 768px) {
            .cards-grid {
                grid-template-columns: 1fr;
            }
            .stats-card {
                grid-template-columns: 1fr;
            }
            .mobile-nav-toggle {
                display: flex;
            }
            .sidebar {
                transform: translateX(-100%);
                width: 250px;
            }
            .sidebar.active {
                transform: translateX(0);
            }
            .menu-item span, .section-title {
                display: block;
            }
            .menu-item {
                justify-content: flex-start;
                padding: 12px 20px;
            }
            .menu-item i {
                margin-right: 10px;
            }
            .main-content {
                margin-left: 0;
                width: 100%;
                padding-top: 60px;
            }
            .overlay.active {
                display: block;
            }
        }

        @media (max-width: 576px) {
            .overview-header {
                flex-direction: column;
                align-items: flex-start;
            }
            .view-users-btn {
                margin-top: 10px;
            }
        }
    </style>
</head>
<body>
    <!-- Mobile menu toggle -->
    <button class="mobile-nav-toggle" id="navToggle">
        <i class="fas fa-bars"></i>
    </button>

      <!-- Overlay for closing the sidebar on mobile -->
    <div class="overlay" id="overlay"></div>

    <!-- Sidebar -->
    <div class="sidebar" id="sidebar">
        <div class="sidebar-header">ADMIN DASHBOARD</div>
        <div class="sidebar-menu">
            <a href="/admin/admin_dashboard.php" class="menu-item active">
                <i class="fas fa-th-large"></i>
                <span>Dashboard</span>
            </a>
            <a href="/admin/coin-deposit-request.php" class="menu-item">
                <i class="fas fa-arrow-circle-down"></i>
                <span>Deposit Coin</span>
            </a>
             <a href="/admin/deposit_requests.php" class="menu-item">
                <i class="fas fa-money-check-alt"></i>
                <span>Deposit Check</span>
            </a>
            <a href="/admin/transfer-request.php" class="menu-item">
                <i class="fas fa-arrow-circle-up"></i>
                <span>Withdraw</span>
            </a>
            <a href="/admin/manage-payment-method.php" class="menu-item">
                <i class="fas fa-exchange-alt"></i>
                <span>Manage Payment Method</span>
            </a>
            <a href="/admin/admin_staking.php" class="menu-item">
                <i class="fas fa-lock"></i>
                <span>Manage Staking</span>
            </a>
            <a href="/admin/admin_staking_transactions.php" class="menu-item">
                <i class="fas fa-lock"></i>
                <span>Manage Staking Transations</span>
            </a>
            <a href="/admin/admin_stocks.php" class="menu-item">
                <i class="fas fa-chart-bar"></i>
                <span>Manage Stocks</span>
            </a>
            <a href="/admin/admin_stock_transactions.php" class="menu-item">
                <i class="fas fa-chart-bar"></i>
                <span>Manage Stocks Transations</span>
            </a>
            <a href="/admin/admin_bots.php" class="menu-item">
                <i class="fas fa-robot"></i>
                <span>Manage Bot</span>
            </a>
            <a href="/admin/investment.php" class="menu-item">
                <i class="fas fa-coins"></i>
                <span>Manage Investment</span>
            </a>
            
            <a href="/admin/crypto_orders.php" class="menu-item">
                <i class="fas fa-coins"></i>
                <span>Manage Crypto Order</span>
            </a>
            
            <a href="/admin/update_balance.php" class="menu-item">
                <i class="fas fa-file-invoice-dollar"></i>
                <span>Update Balances</span>
            </a>
            <a href="/admin/kyc-request.php" class="menu-item">
                <i class="fas fa-id-card"></i>
                <span>KYC</span>
            </a>
            <a href="/admin/admin_send_notification.php" class="menu-item">
                <i class="fas fa-bell"></i>
                <span>Manage Notifications</span>
            </a>

            <div class="section-title">USER MANAGEMENT</div>
            <a href="/admin/manage_users.php" class="menu-item">
                <i class="fas fa-users"></i>
                <span>Manage Users</span>
            </a>
            <div class="section-title">Approve SignUp</div>
            <a href="/admin/admin_approval.php" class="menu-item">
                <i class="fas fa-user-check"></i>
                <span>Approve SignUp</span>
            </a>
            <div class="section-title">Change Password</div>
            <a href="/admin/change_password.php" class="menu-item">
                <i class="fas fa-unlock-alt"></i>
                <span>Change Password</span>
            </a>

            <div class="section-title">MANAGEMENT</div>
            <a href="/admin/logout.php" class="menu-item">
                <i class="fas fa-sign-out"></i>
                <span>Sign Out</span>
            </a>
        </div>
    </div>



    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <div class="top-bar">
            <div class="page-icon">
                <i class="fas fa-clipboard"></i>
            </div>
            <div class="user-info">
                <div class="user-icon">
                    <i class="fas fa-user"></i>
                </div>
                <div class="user-role">
                    <div>Admin</div>
                    <div class="role-name">Administrator <i class="fas fa-chevron-down"></i></div>
                </div>
            </div>
        </div>

        <!-- Overview Section -->
        <div class="overview-section">
            <div class="overview-header">
                <div>
                    <h2 class="overview-title">Overview</h2>
                    <p class="overview-description">Here is an insight of what's going on.</p>
                </div>
                <button class="view-users-btn">
                    <i class="fas fa-users"></i> View All Users
                </button>
            </div>

    <style>
        .admin-container { max-width: 1200px; margin: 0 auto; padding: 20px; }
        .action-btn { padding: 5px 10px; border: none; border-radius: 4px; cursor: pointer; margin-right: 5px; }
        .approve-btn { background: #4cc9f0; color: white; }
        .reject-btn { background: #f72585; color: white; }
        .edit-btn { background: #4895ef; color: white; }
        .delete-btn { background: #f72585; color: white; }
        table { width: 100%; border-collapse: collapse; margin: 20px 0; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background-color: #f2f2f2; }
        .form-group { margin-bottom: 15px; }
        label { display: block; margin-bottom: 5px; }
        input, textarea, select { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; }
        .btn { padding: 10px 15px; background: #4895ef; color: white; border: none; border-radius: 4px; cursor: pointer; }
        .btn-danger { background: #f72585; }
        .notification { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .success { background: #4cc9f0; color: white; }
        .error { background: #f72585; color: white; }
        .tabs { display: flex; margin-bottom: 20px; }
        .tab { padding: 10px 20px; cursor: pointer; border: 1px solid #ddd; }
        .tab.active { background: #4895ef; color: white; }
        .tab-content { display: none; }
        .tab-content.active { display: block; }
    </style>

    <div class="admin-container">
    <h1>Bot Management System</h1>
    
    <?php if (isset($_SESSION['success_message'])): ?>
        <div class="notification success"><?= $_SESSION['success_message'] ?></div>
        <?php unset($_SESSION['success_message']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error_message'])): ?>
        <div class="notification error"><?= $_SESSION['error_message'] ?></div>
        <?php unset($_SESSION['error_message']); ?>
    <?php endif; ?>
    
    <div class="tabs">
        <div class="tab active" onclick="switchTab('approvals')">Pending Approvals</div>
        <div class="tab" onclick="switchTab('manage')">Manage Bots</div>
    </div>
    
    <div id="approvals" class="tab-content active">
        <h2>Pending Bot Approvals</h2>
        
        <?php if (empty($pendingBots)): ?>
            <p>No pending approvals</p>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>User</th>
                        <th>Bot</th>
                        <th>Type</th>
                        <th>Amount</th>
                        <th>Return Rate</th>
                        <th>Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($pendingBots as $bot): ?>
                        <tr>
                            <td><?= htmlspecialchars($bot['username']) ?></td>
                            <td><?= htmlspecialchars($bot['bot_name']) ?></td>
                            <td><?= strtoupper($bot['bot_type']) ?></td>
                            <td>$<?= number_format($bot['amount'], 2) ?></td>
                            <td><?= isset($bot['return_rate']) ? number_format($bot['return_rate'], 2).'%' : 'N/A' ?></td>
                            <td><?= date('M d, Y H:i', strtotime($bot['created_at'])) ?></td>
                            <td>
                                <button class="action-btn approve-btn" onclick="approveTransaction(<?= $bot['id'] ?>)">
                                    <i class="fas fa-check"></i> Approve
                                </button>
                                <button class="action-btn reject-btn" onclick="rejectTransaction(<?= $bot['id'] ?>)">
                                    <i class="fas fa-times"></i> Reject
                                </button>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
    
    <div id="manage" class="tab-content">
        <h2><?= $editBot ? 'Edit' : 'Create' ?> Trading Bot</h2>
        
        <form method="POST">
            <input type="hidden" name="bot_action" value="<?= $editBot ? 'update' : 'create' ?>">
            <?php if ($editBot): ?>
                <input type="hidden" name="id" value="<?= $editBot['id'] ?>">
            <?php endif; ?>
            
            <div class="form-group">
                <label for="name">Bot Name</label>
                <input type="text" id="name" name="name" required 
                       value="<?= htmlspecialchars($editBot['name'] ?? '') ?>">
            </div>
            
            <div class="form-group">
                <label for="type">Bot Type</label>
                <select id="type" name="type" required>
                    <option value="crypto" <?= isset($editBot) && $editBot['type'] === 'crypto' ? 'selected' : '' ?>>Crypto</option>
                    <option value="forex" <?= isset($editBot) && $editBot['type'] === 'forex' ? 'selected' : '' ?>>Forex</option>
                    <option value="stocks" <?= isset($editBot) && $editBot['type'] === 'stocks' ? 'selected' : '' ?>>Stocks</option>
                </select>
            </div>
            
            <div class="form-group">
                <label for="description">Description</label>
                <textarea id="description" name="description" required><?= htmlspecialchars($editBot['description'] ?? '') ?></textarea>
            </div>
            
            <div class="form-group">
                <label for="symbol">Trading Symbol</label>
                <input type="text" id="symbol" name="symbol" required
                       value="<?= htmlspecialchars($editBot['symbol'] ?? '') ?>">
            </div>
            
<div class="form-group">
    <label for="minimum_amount">Minimum Amount (USD)</label>
    <input type="number" id="minimum_amount" name="minimum_amount" step="0.01" min="0" required 
           value="<?= htmlspecialchars($editBot['minimum_amount'] ?? '') ?>">
</div>
            
            <div class="form-group">
                <label for="maximum_amount">Maximum Amount (USD)</label>
                <input type="number" id="maximum_amount" name="maximum_amount" step="0.01" min="0" required 
                       value="<?= htmlspecialchars($editBot['maximum_amount'] ?? '') ?>">
            </div>
            
            <div class="form-group">
                <label for="return_rate">Return Rate (%)</label>
                <input type="number" id="return_rate" name="return_rate" step="0.01" required 
                       value="<?= htmlspecialchars($editBot['return_rate'] ?? '') ?>">
            </div>
            
            <div class="form-group">
                <label for="cycle">Cycle (days)</label>
                <input type="number" id="cycle" name="cycle" min="1" required 
                       value="<?= htmlspecialchars($editBot['cycle'] ?? '30') ?>">
            </div>
            
            <button type="submit" class="btn">
                <?= $editBot ? 'Update Bot' : 'Create Bot' ?>
            </button>
            
            <?php if ($editBot): ?>
                <a href="admin_bots.php" class="btn">Cancel</a>
            <?php endif; ?>
        </form>
        
        <h2>All Trading Bots</h2>
        
        <?php if (empty($allBots)): ?>
            <p>No bots created yet</p>
        <?php else: ?>
            <table>
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Type</th>
                        <th>Symbol</th>
                        <th>Min Amount</th>
                        <th>Max Amount</th>
                        <th>Return Rate</th>
                        <th>Cycle</th>
                        <th>Created</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($allBots as $bot): ?>
                        <tr>
                            <td><?= htmlspecialchars($bot['name']) ?></td>
                            <td><?= ucfirst($bot['type']) ?></td>
                            <td><?= $bot['symbol'] ?></td>
                            <td>$<?= number_format($bot['minimum_amount'], 2) ?></td>
                            <td>$<?= number_format($bot['maximum_amount'], 2) ?></td>
                            <td><?= number_format($bot['return_rate'], 2) ?>%</td>
                            <td><?= $bot['cycle'] ?> days</td>
                            <td><?= date('M d, Y', strtotime($bot['created_at'])) ?></td>
                            <td>
                                <a href="admin_bots.php?edit=<?= $bot['id'] ?>" class="action-btn edit-btn">
                                    <i class="fas fa-edit"></i> Edit
                                </a>
                                <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this bot?')">
                                    <input type="hidden" name="bot_action" value="delete">
                                    <input type="hidden" name="id" value="<?= $bot['id'] ?>">
                                    <button type="submit" class="action-btn delete-btn">
                                        <i class="fas fa-trash"></i> Delete
                                    </button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</div>
    <script>
      // Get references to elements
        const navToggle = document.getElementById('navToggle');
        const sidebar = document.getElementById('sidebar');
        const overlay = document.getElementById('overlay');
        const mainContent = document.getElementById('mainContent');

        // Toggle sidebar
        navToggle.addEventListener('click', function() {
            sidebar.classList.toggle('active');
            overlay.classList.toggle('active');
        });

        // Close sidebar when clicking on overlay
        overlay.addEventListener('click', function() {
            sidebar.classList.remove('active');
            overlay.classList.remove('active');
        });

        // Close sidebar when clicking on main content (for mobile)
        mainContent.addEventListener('click', function() {
            if (window.innerWidth <= 768 && sidebar.classList.contains('active')) {
                sidebar.classList.remove('active');
                overlay.classList.remove('active');
            }
        });

        // Handle window resize
        window.addEventListener('resize', function() {
            if (window.innerWidth > 768) {
                sidebar.classList.remove('active');
                overlay.classList.remove('active');
            }
        });

        // Close sidebar when clicking menu items (for mobile)
        const menuItems = document.querySelectorAll('.menu-item');
        menuItems.forEach(item => {
            item.addEventListener('click', function() {
                if (window.innerWidth <= 768) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                }
            });
        });
        // Tab switching
        function switchTab(tabId) {
            document.querySelectorAll('.tab').forEach(tab => {
                tab.classList.remove('active');
            });
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });
            
            document.querySelector(`.tab[onclick="switchTab('${tabId}')"]`).classList.add('active');
            document.getElementById(tabId).classList.add('active');
        }
        
        // Transaction approval functions
        async function approveTransaction(id) {
            if (!confirm('Approve this bot launch?')) return;
            
            try {
                const response = await fetch('admin_bots.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        action: 'approve',
                        transaction_id: id
                    })
                });
                
                const result = await response.json();
                
                if (response.ok && result.success) {
                    alert('Bot approved successfully!');
                    location.reload();
                } else {
                    throw new Error(result.error || 'Approval failed');
                }
            } catch (error) {
                alert(error.message);
            }
        }
        
        async function rejectTransaction(id) {
            if (!confirm('Reject this bot launch and refund user?')) return;
            
            try {
                const response = await fetch('admin_bots.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        action: 'reject',
                        transaction_id: id
                    })
                });
                
                const result = await response.json();
                
                if (response.ok && result.success) {
                    alert('Bot rejected and user refunded!');
                    location.reload();
                } else {
                    throw new Error(result.error || 'Rejection failed');
                }
            } catch (error) {
                alert(error.message);
            }
        }
    </script>
</body>
</html>