<?php
session_start();

// Enable error reporting
ini_set('display_errors', 1);
error_reporting(E_ALL);

// Database connection using PDO
include($_SERVER['DOCUMENT_ROOT'] . '/db_config.php');

// Function to handle errors and set session messages
function showError($message) {
    $_SESSION['error_message'] = $message;
    // Return an HTML-formatted error message
    return "<div class='error-message'><i class='fas fa-exclamation-circle'></i> $message</div>";
}

// Include PHPMailer
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

// Check if the request is a POST request
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Create database connection using config.php credentials
        $conn = new PDO("mysql:host=$host;dbname=$dbname", $username, $password);
        $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        // Sanitize inputs
        $username_input = htmlspecialchars(trim($_POST['username']));
        $full_name = htmlspecialchars(trim($_POST['full_name']));
        $email = filter_var(trim($_POST['email']), FILTER_SANITIZE_EMAIL);
        $phone = trim($_POST['phone']);
        $address = htmlspecialchars(trim($_POST['address']));
        $password_input = $_POST['password'];
        $confirm_password = $_POST['confirm_password'];

        // Validate email
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            echo showError("Invalid email format.");
            exit;
        }

        // Accept any phone number format (removed strict validation)
        // Just ensure it's not empty
        if (empty($phone)) {
            echo showError("Phone number cannot be empty.");
            exit;
        }

        // Check if password and confirm password match
        if ($password_input !== $confirm_password) {
            echo showError("Passwords do not match!");
            exit;
        }

        // Hash the password
        $hashed_password = password_hash($password_input, PASSWORD_DEFAULT);

        // Generate a unique 10-digit account number
        $account_number = mt_rand(1000000000, 9999999999);

        // Generate a unique 6-digit verification code
        $verification_code = mt_rand(100000, 999999);

        // Check if email or username already exists
        $stmt = $conn->prepare("SELECT * FROM users WHERE email = :email OR username = :username");
        $stmt->bindParam(':email', $email);
        $stmt->bindParam(':username', $username_input);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            echo showError("Username or email already taken.");
            exit;
        }

        // Insert user data
        $stmt = $conn->prepare("INSERT INTO users (username, full_name, email, phone, address, password, verification_code, account_number, created_at) 
                                VALUES (:username, :full_name, :email, :phone, :address, :password, :verification_code, :account_number, NOW())");

        $stmt->bindParam(':username', $username_input);
        $stmt->bindParam(':full_name', $full_name);
        $stmt->bindParam(':email', $email);
        $stmt->bindParam(':phone', $phone);
        $stmt->bindParam(':address', $address);
        $stmt->bindParam(':password', $hashed_password);
        $stmt->bindParam(':verification_code', $verification_code);
        $stmt->bindParam(':account_number', $account_number);

        // Execute the query
        if ($stmt->execute()) {
            // Include the send_email.php file
            include('send_email.php');

            // Call the sendVerificationEmail function
            if (sendVerificationEmail($email, $verification_code)) {
                $_SESSION['email'] = $email;
                header("Location: email_verification.php");
                exit;
            } else {
                echo showError("Failed to send verification email.");
            }
        } else {
            echo showError("Failed to create user. Please try again later.");
        }
    } catch (PDOException $e) {
        echo showError("Database error: " . $e->getMessage());
    } catch (Exception $e) {
        echo showError("Error: " . $e->getMessage());
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sign Up - Grownest Capital</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --bg-primary: #1c1c28;
            --bg-secondary: #23232f;
            --text-primary: #ffffff;
            --text-secondary: #a0a0a0;
            --accent-color: #6366f1;
            --border-color: #32323f;
            --menu-hover: #32323f;
            --card-bg: #23232f;
            --error-color: #ff6b6b;
            --success-color: #4CAF50;
        }

        .light-mode {
            --bg-primary: #f5f5f9;
            --bg-secondary: #ffffff;
            --text-primary: #333333;
            --text-secondary: #666666;
            --accent-color: #6366f1;
            --border-color: #e0e0e0;
            --menu-hover: #e8e8f0;
            --card-bg: #ffffff;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        body {
            background-color: var(--bg-primary);
            color: var(--text-primary);
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            padding: 20px;
        }

        .signup-container {
            background-color: var(--card-bg);
            border-radius: 12px;
            padding: 40px;
            border: 1px solid var(--border-color);
            width: 100%;
            max-width: 500px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
        }

        .logo {
            text-align: center;
            margin-bottom: 30px;
        }

        .logo img {
            width: 60px;
            height: 60px;
            margin-bottom: 15px;
        }

        .logo h2 {
            font-size: 1.5rem;
            color: var(--text-primary);
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            color: var(--text-secondary);
            font-size: 0.9rem;
            font-weight: 500;
        }

        .form-control {
            width: 100%;
            padding: 12px 15px;
            border-radius: 8px;
            border: 1px solid var(--border-color);
            background-color: var(--bg-primary);
            color: var(--text-primary);
            font-size: 1rem;
            transition: border-color 0.2s ease;
        }

        .form-control:focus {
            outline: none;
            border-color: var(--accent-color);
        }

        textarea.form-control {
            min-height: 100px;
            resize: vertical;
        }

        .password-container {
            position: relative;
        }

        .password-toggle {
            position: absolute;
            right: 15px;
            top: 42px;
            transform: translateY(-50%);
            color: var(--text-secondary);
            cursor: pointer;
        }

        .signup-btn {
            width: 100%;
            padding: 12px;
            background-color: var(--accent-color);
            color: white;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            font-size: 1rem;
            cursor: pointer;
            transition: background-color 0.2s ease;
            margin-top: 10px;
        }

        .signup-btn:hover {
            background-color: #4f46e5;
        }

        .form-footer {
            text-align: center;
            margin-top: 25px;
            color: var(--text-secondary);
            font-size: 0.9rem;
        }

        .form-footer a {
            color: var(--accent-color);
            text-decoration: none;
            font-weight: 600;
        }

        .form-footer a:hover {
            text-decoration: underline;
        }

        .password-strength {
            margin-top: 5px;
            font-size: 0.8rem;
            color: var(--text-secondary);
        }

        /* Error Message Styling */
        .error-message {
            background-color: rgba(255, 107, 107, 0.15);
            color: var(--error-color);
            padding: 12px 16px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-size: 0.95rem;
            border-left: 4px solid var(--error-color);
            display: flex;
            align-items: center;
            animation: fadeIn 0.3s ease-in-out;
        }

        .error-message i {
            margin-right: 10px;
            font-size: 1.1rem;
        }

        /* Success Message Styling */
        .success-message {
            background-color: rgba(76, 175, 80, 0.15);
            color: var(--success-color);
            padding: 12px 16px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-size: 0.95rem;
            border-left: 4px solid var(--success-color);
            display: flex;
            align-items: center;
            animation: fadeIn 0.3s ease-in-out;
        }

        .success-message i {
            margin-right: 10px;
            font-size: 1.1rem;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        @media (max-width: 576px) {
            .signup-container {
                padding: 30px 20px;
            }
        }
    </style>
</head>
<body>
    <div class="signup-container">
        <div class="logo">
            <img src="/uploads/output.png" alt="Pocket Wallet Logo">
            <h2>Create Your Account</h2>
        </div>

        <?php
        // Display error message if exists in session
        if (isset($_SESSION['error_message'])) {
            echo "<div class='error-message'><i class='fas fa-exclamation-circle'></i> ".$_SESSION['error_message']."</div>";
            unset($_SESSION['error_message']); // Clear the message after displaying
        }
        ?>

        <form action="signup.php" method="POST">
            <div class="form-group">
                <label for="username">Username</label>
                <input type="text" id="username" name="username" class="form-control" placeholder="Enter your username" required>
            </div>

            <div class="form-group">
                <label for="full_name">Full Name</label>
                <input type="text" id="full_name" name="full_name" class="form-control" placeholder="Enter your full name" required>
            </div>

            <div class="form-group">
                <label for="email">Email Address</label>
                <input type="email" id="email" name="email" class="form-control" placeholder="Enter your email" required>
            </div>

            <div class="form-group">
                <label for="phone">Phone Number</label>
                <input type="text" id="phone" name="phone" class="form-control" placeholder="Enter your phone number" required>
            </div>

            <div class="form-group">
                <label for="address">Address</label>
                <textarea id="address" name="address" class="form-control" placeholder="Enter your address" required></textarea>
            </div>

            <div class="form-group">
                <label for="password">Password</label>
                <div class="password-container">
                    <input type="password" id="password" name="password" class="form-control" placeholder="Enter your password" required>
                    <i class="fas fa-eye password-toggle" id="togglePassword"></i>
                </div>
                <div class="password-strength">Use 8 or more characters with a mix of letters, numbers & symbols</div>
            </div>

            <div class="form-group">
                <label for="confirm_password">Confirm Password</label>
                <div class="password-container">
                    <input type="password" id="confirm_password" name="confirm_password" class="form-control" placeholder="Confirm your password" required>
                    <i class="fas fa-eye password-toggle" id="toggleConfirmPassword"></i>
                </div>
            </div>

            <button type="submit" name="signup" class="signup-btn">
                Create Account <i class="fas fa-arrow-right" style="margin-left: 5px;"></i>
            </button>

            <div class="form-footer">
                <p>Already have an account? <a href="login.php">Sign In</a></p>
            </div>
        </form>
    </div>

    <script>
        // Toggle password visibility
        const togglePassword = document.querySelector('#togglePassword');
        const password = document.querySelector('#password');
        const toggleConfirmPassword = document.querySelector('#toggleConfirmPassword');
        const confirmPassword = document.querySelector('#confirm_password');

        togglePassword.addEventListener('click', function() {
            const type = password.getAttribute('type') === 'password' ? 'text' : 'password';
            password.setAttribute('type', type);
            this.classList.toggle('fa-eye-slash');
        });

        toggleConfirmPassword.addEventListener('click', function() {
            const type = confirmPassword.getAttribute('type') === 'password' ? 'text' : 'password';
            confirmPassword.setAttribute('type', type);
            this.classList.toggle('fa-eye-slash');
        });

        // Theme toggle functionality if needed
        const themeToggle = document.getElementById('theme-toggle');
        if (themeToggle) {
            themeToggle.addEventListener('click', () => {
                document.body.classList.toggle('light-mode');
            });
        }
    </script>
</body>
</html>